define(['app', 'angular'], function (app) {
	'use strict';

	app.directive('inputRange', function () {
		return {
			restrict: 'E',
			require: '^form',
			scope: {
				'formName': '@',  //required(used for custom validation)
				'label': '@',
				'name': '@',
				'ngDisabled': '=',
				'ngRequired': '=',
				'ngReadonly': '=',
				'min': '@',
				'max': '@',
				'ngModel': '=',
				'minLabel': '@',
				'maxLabel': '@',
				'step': '@',
				isSubLabel : '@'
			},
			link: function (scope, element, attrs, ngModelCtrl){
				scope.$watch('ngModel', function (newVal, oldVal) {
					if (newVal !== oldVal) {
						ngModelCtrl.$setDirty();
					}
				});
			},
			controller: function ($scope, $window, $element, $swipe, $timeout) {
				var THUMB_WIDTH = 40,
					angularizedInputElement = $element.find("input"),
					inputElement = angularizedInputElement[0],
					inputElementPosition = inputElement.getBoundingClientRect(), //bottom, height, left, right, top, width
					inSwipeEvent = false,
					minVal = parseFloat($scope.min),
					maxVal = parseFloat($scope.max),
					range = maxVal - minVal;

				$scope.inputModel = {"value": $scope.ngModel || $scope.min};
				$scope.rangeValues = getRangeValues();

				function getRangeValues() {
					var values = [];
					for (var i = minVal; i <= maxVal; i++) {
						values.push(i);
					}
					return values;
				}

				(function estimatedPositions() {
					var elementWidth = $element.width();
					$scope.thumbDisplacement = {'left': 100 * (1 - (THUMB_WIDTH / elementWidth)) * (parseFloat($scope.inputModel.value) - minVal) / range + "%"};
				})();

				$scope.iOS = ( navigator.userAgent.match(/(iPad|iPhone|iPod)/g) ? true : false );
				$scope.liveLabel = {'inputValue': $scope.inputModel.value,
					'labelReadable': false,
					'valueReadable': false};

				$scope.updateNgModelAndSlider = function () {
					$timeout(function () {
						$scope.ngModel = $scope.inputModel.value;
						$scope.formName && $scope.$parent[$scope.formName][$scope.name].$setViewValue($scope.inputModel.value);

						if (!inSwipeEvent) {
							updateSliderWithModel();
						}
						$scope.liveLabel.labelReadable = false;
						$scope.liveLabel.valueReadable = true;
					});
				};

				$scope.$watch('ngModel', function (newVal, oldVal) {
					if (newVal !== oldVal) {
						$scope.inputModel.value = $scope.ngModel || $scope.min;
						updateSliderWithModel();
					}

					$scope.formName && $scope.$parent[$scope.formName][$scope.name].$setValidity($scope.name, newVal !== undefined && newVal !== null);
				});

				$scope.$watch('parseFloat(max) - parseFloat(min)', function (newVal, oldVal) {
					if (newVal !== oldVal) {
						range = newVal;
						$scope.$apply(function () {
							minVal = parseFloat($scope.min);
							maxVal = parseFloat($scope.max);
							updateSliderWithModel();
						});
					}
				});

				$scope.$watch(function () {
					return $element.find("input").is(':visible');
				}, function (isVisible) {
					if (isVisible) {
						$timeout(function () {
							inputElementPosition = inputElement.getBoundingClientRect();
							updateSliderWithModel();
						});
					}
				});

				$scope.$watch(function () {
					return inputElement.getBoundingClientRect().left;
				}, function (currentLeft, prevLeft) {
					if (currentLeft !== prevLeft) {
						$timeout(function () {
							inputElementPosition = inputElement.getBoundingClientRect();
							updateSliderWithModel();
						});
					}
				});

				angularizedInputElement.bind("focus", function () {
					$scope.liveLabel.inputValue = $scope.inputModel.value;
					$scope.liveLabel.labelReadable = true;
					$scope.liveLabel.valueReadable = true;
				});

				angularizedInputElement.bind("blur", function () {
					$scope.liveLabel.labelReadable = false;
					$scope.liveLabel.valueReadable = false;
				});

				$element.bind('resize', function () {
					$timeout(function () {
						inputElementPosition = inputElement.getBoundingClientRect();
						updateSliderWithModel();
					});
				});

				(function bindSwipe() {
					function updateModelAndSlider(withPos) {
						$timeout(function () {
							var percentLeft = (withPos.x - inputElementPosition.left) / inputElementPosition.width;
							if (percentLeft > 1) {
								percentLeft = 1;
							} else if (percentLeft < 0) {
								percentLeft = 0;
							}

							$scope.inputModel.value = (Math.round(range * percentLeft) + minVal).toString();
							$scope.updateNgModelAndSlider();

							$scope.thumbDisplacement.left = 100 * percentLeft*(1-(THUMB_WIDTH/inputElementPosition.width)) + "%";
						});
					}

					function onStart(pos) {
						inSwipeEvent = true;
						updateModelAndSlider(pos);
					}

					function onEnd() {
						inSwipeEvent = false;
						$scope.updateNgModelAndSlider();
					}

					function onMove(pos) {
						updateModelAndSlider(pos);
					}

					$swipe.bind(angularizedInputElement, {
						start: onStart,
						move: onMove,
						end: onEnd,
						cancel: onEnd
					});
				})();

				function updateSliderWithModel() {
					if (!inSwipeEvent && inputElementPosition.width !== 0) {
						var percentLeft = (parseFloat($scope.inputModel.value) - minVal) / range;
						$scope.thumbDisplacement.left = 100 * percentLeft * (1-(THUMB_WIDTH/inputElementPosition.width)) + "%";
					}
				}
			},
			templateUrl: 'src/ui-components/form/controls/simple/input-range/input-range_template.html'
		};
	});
});
